//
//  rtfbp1.cpp
//  rtfbp1
//
//  Created by Stanko Juzbasic on 1/19/16.
//
//  this is the enhanced Float Buffer Player program soruce code, with redirection & piping
//  signals from the keyboard are being caught and handled for graceful program termination
//  Written in RTaudio API

#include "RtAudio.h"
#include <iostream>
#include <cstdlib>
#include <cmath>
#include <cstring>
#include <unistd.h>
#include <sys/select.h>

#include <sys/types.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/time.h>

//____declarations_______________________________________________________________________
// globals
unsigned int sampleRate = 44100;

RtAudio dac;

typedef struct {
    double  j;
    unsigned int   N;
    unsigned int   totalFrames;
    double         playedFrames;    
    float*         fBuffer;
    size_t         fbuffersize;
    bool           START;
    bool           STOP;
    bool           END;
    int            M;
}uData;

volatile sig_atomic_t s_flag = 0;

//Prototypes______________________________________________________________________________
void usage (void);

void prompt1 (void);

void sig_handler(int);

bool rt_sig_timed_kbhit(int, struct timeval);

double kramp(double);

size_t getSize(FILE* , char* );

struct timeval get_duration (uData*, double);

void normalize(float* , size_t , float );

int fbrp(void*, void*, unsigned int, double, RtAudioStreamStatus, void* ); //rendering
//________________________________________________________________________________________
#pragma mark - auxilliary function(s) -
//________________________________________________________________________________________
void usage (void)
{
    std::cout << "\nuseage: rtfbp1 fs file <device> <channelOffset>\n";
    std::cout << "    fs = the sample rate, \n";
    std::cout << "    file = the raw file to play, 0 = stdin\n";
    std::cout << "    device = optional device to use (default = 0),\n";
    std::cout << "    and channelOffset = an optional channel offset on the device (default = 0).\n\n";
    return;
}
//________________________________________________________________________________________
void prompt1 (void)
{
    std::cout << "Press ";
    std::cout <<"<enter>";
    std::cout << " to stop."<<std::endl;
}
//________________________________________________________________________________________
// this is a UNIX signal handler for ^C, ^Z, ^\
//________________________________________________________________________________________
void sig_handler(int signo)
{
    switch (signo) {
        case SIGINT:
            std::cout<<"\nreceived SIGINT"<<std::endl;
            s_flag = SIGINT;//handle SIGINT
            break;
        case SIGTSTP:
            std::cout<<"\nreceived SIGTSTP"<<std::endl;
            s_flag = SIGTSTP;//handle SIGTSTP
            break;
        case SIGQUIT:
            std::cout<<"\nreceived SIGQUIT"<<std::endl;
            s_flag = SIGUSR1;//handle SIGQUIT
            break;
        case SIGUSR1:
            std::cout<<"\nreceived SIGUSR1"<<std::endl;
            s_flag = SIGUSR1;//handle SIGUSR1
            break;
        default:
            break;
    }
}
//________________________________________________________________________________________
// a UNIX function for non-blocking keyboard hit detector and signal catcher, RTaudio
//________________________________________________________________________________________
bool rt_sig_timed_kbhit(int fd, struct timeval tv, uData * data)
{
    // Install the signal handler for several SIGNALS
    struct sigaction s;
    s.sa_handler = sig_handler;
    sigemptyset(&s.sa_mask);
    s.sa_flags = 0;
    sigaction(SIGINT,  &s, NULL);
    sigaction(SIGTSTP, &s, NULL);
    sigaction(SIGQUIT, &s, NULL);
    
    // Block several SIGNALS
    sigset_t sigset, prev_set;
    sigemptyset(&sigset);
    sigaddset(&sigset, SIGINT);
    sigaddset(&sigset, SIGTSTP);
    sigaddset(&sigset, SIGQUIT);
    sigprocmask(SIG_BLOCK, &sigset, &prev_set);
    
    struct timespec ts;
    TIMEVAL_TO_TIMESPEC(&tv, &ts);
    fd_set read_fd;
    FD_ZERO(&read_fd);
    FD_SET(fd, &read_fd);
    
    while(!(data->START));
    
    if (pselect(fd+1, &read_fd, NULL, NULL, &ts, &prev_set) < 0)
        std::cerr<<"select"<<std::endl;

    bool   b = (FD_ISSET(fd, &read_fd)) ; 
    return b;
}
//________________________________________________________________________________________
size_t getSize(FILE* file, char* name)
{
    size_t s;
    if(file == NULL)return 0;
    fseek(file, 0L, SEEK_END);
    s = ftell(file);
    s = s/sizeof(float);
    std::cout<<"File "<<name<<": "<<s<<"\t values\n"<<std::endl;
    rewind(file);
    return s;
}
//________________________________________________________________________________________
struct timeval get_duration (uData *data, double sRate)
{
    struct timeval tv;
    tv.tv_sec =(time_t)(data->fbuffersize/sRate);
    tv.tv_usec=(int)(1000000*((data->fbuffersize/sRate) - tv.tv_sec));
    std::cout<<"Duration: "<<tv.tv_sec<<"."<<tv.tv_usec<<"s"<<std::endl;
    return tv;
}
//________________________________________________________________________________________
void normalize(float* buf, size_t siz, float norm_max)
{
    float m;
    if(norm_max > .99f)norm_max = .99f;
    for (int i = 0; i < siz; i++)
        if (fabsf(buf[i]) > m ) {
            m = fabsf(buf[i]);
        }
    m = norm_max/m;
    for (int j = 0 ; j < siz; j++)
        buf[j] = m*buf[j];
    
    return;
}
//___________________________________________________________________________________
// generate 1000 ramp samples 
double kramp ( double i)
{
    if(i >= 1000.)return 1.;
    if(i <= 0.)   return 0.;
    return .001*(i);
}
//____________________________________________________________________________________
#pragma mark - callback -
//____________________________________________________________________________________
// Two-channel float buffer player with start/stop,v.1
int fbrp(void *outputBuffer,
         void *inputBuffer,
         unsigned int nBufferFrames,
         double streamTime,
         RtAudioStreamStatus status,
         void *userData )
{
    static int stopped = 1;
    
    double *buffer = (double *) outputBuffer;

    uData *data = (uData*)userData;
    
    unsigned int frame;
    double  d;
    long   l = (long)data->j;
    
    if ( status )
        std::cout << "Stream underflow detected!" << std::endl;
    
    if((unsigned int)(data->totalFrames)  >= (data->fbuffersize - 2*nBufferFrames)) {
        if(0)std::cout<<"END"<<std::endl;
        data->END = true;
    }
    
    // Writing interleaved stereo audio data from float buffer
    for (frame = 0; frame < nBufferFrames; ++frame)
	{
        d = 0.;

        //compute samples data according to conditions
        if(data->totalFrames >= data->fbuffersize){
            d =  0.;
        }else if(data->STOP){
            if(data->N < 1)data->N = 1;
            d =  data->fBuffer[l]*kramp((double)data->N--);
        }else if(data->END){
            if((data->totalFrames < data->fbuffersize) && (stopped)){
                std::cout<<"end sequence @"<<data->totalFrames<<std::endl;;
                stopped = 0;
            }
            double aa = kramp(data->M);
            d =  data->fBuffer[l]*aa;            
            data->M = data->M - 3; 
        }else if(data->totalFrames < 1000){
            if(data->START == false)data->START = true;
            double a = kramp((int)(data->totalFrames));
            d =  data->fBuffer[l]*a;
        }else{
            d =  data->fBuffer[l];
        }
        //copy data to buffer
        unsigned int fr = frame+frame;
        buffer[fr+1] = buffer[fr] = (double)d;

		data->j += 1.0;
        data->totalFrames +=1;
        l++;
        if(0)std::cout<<data->N<<std::endl;
	}

    if(data->totalFrames >= data->fbuffersize){
        if(0)std::cout<<"game over!"<<std::endl;
        //signal the stream to stop by passing a non-zero return value
        return 1;
    }
    return 0;
}
//____________________________________________________________________________________
#pragma mark - main -
//____________________________________________________________________________________
int main(int argc, const char * argv[])
{
    FILE *afPtr;
    char afname[512];
    size_t size;
    float *temp, val;
    bool STD = false;
    
    unsigned int  fs, bufferFrames, device = 0, offset = 0;

    uData data;
    
    data.N = 2000; data.STOP = false;
    
    if((argc < 3)||(argc > 5)){
        usage();
        exit (0);
    }
    
    std::cout << "\n_____________________________________________\n";
    std::cout << "* * * Float Buffer Player 1 for RTAudio * * *\n";
    std::cout <<"\n\n";
    
    if(0)    std::cout << argc <<std::endl;
    
    //parsing command-line arguments:
    if ( argc > 4 )
        offset = (unsigned int) atoi( argv[4] );
    
    if ( argc > 3 )
        device = (unsigned int) atoi( argv[3] );
    
    if (argv[2] == NULL){
        goto no_name;        
    }
    else if (strcmp(argv[2],"0") == 0){
        //read from STDIN into temp[]        
        STD = true;
        std::cout<<"Reading from: STDIN"<<std::endl;
        size  = 0;
        afPtr = fdopen(dup(fileno(stdin)), "rb");
        temp  = (float*)malloc(sizeof(float));
        
        while( fread( &val, sizeof(float), 1, afPtr ) == 1){
            temp[size] = val;
            size++;
            temp = (float*)realloc(temp, (size+1)*sizeof(float));
            if(temp == NULL){
                std::cout<<"Realloc: not enough memory!"<<std::endl;
                fclose(afPtr);
                exit(1);
            }
        }
    }// end (strcmp(argv[2],"0")
    else{
        //read from FILE into temp[]
        strcpy(afname, argv[2]);
        std::cout<<"\nName: "<<argv[2]<<"\tFile: "<<afname<<std::endl;
    
        //getting the file, its size and content, allocating buffer:
        afPtr = fopen(afname,"rb");
    
        if (afPtr == NULL){
            std::cout<<"File \""<<argv[2]<<"\"\tnot found"<<std::endl;
    no_name:
            usage();
            exit(0);
        }

        if((size = getSize(afPtr, afname)) <= 0){
            std::cout<<"File "<<afname<<" is empty"<<std::endl;
            fclose(afPtr);
            exit(0);
        };
    
        //allocate temp
        if((temp = (float*)calloc(size + 1, sizeof(float))) == NULL){
            std::cout<<"Not enough memory!"<<std::endl;
            fclose(afPtr);
            exit(0);
        }
    
        //fread() into temp
        fread(temp, sizeof(float), size, afPtr);
        fclose(afPtr);

    }//end reading
    
    data.fbuffersize = size;
    
    data.M = 1000;
        
    if((data.fBuffer  = (float*)calloc(size + 1000, sizeof(float))) == NULL){
        std::cout<<"Not enough memory!"<<std::endl;
        fclose(afPtr);
        exit(0);
    }
    
    fs = (unsigned int) atoi( argv[1] );

    for (int i = 0; i < size; i++){
        data.fBuffer[i] = temp[i];
    if(0)std::cout<<"data.fBuffer["<<i<<"] = "<<data.fBuffer[i]<<std::endl;
    }
    
    normalize(data.fBuffer, size, 0.9f);

    free(temp);
    
    fclose(afPtr);
        
   
    //Open RTaudio stream

    if ( dac.getDeviceCount() < 1 ) {
        std::cout << "\nNo audio devices found!"<<std::endl;
        exit (0);
    }
    
    RtAudio::StreamParameters parameters;
    parameters.deviceId = dac.getDefaultOutputDevice();
    parameters.nChannels = 2;
    parameters.firstChannel = 0;
    data.totalFrames = 0;
    bufferFrames = 512; // 512 sample frames
    
    data.j = 0.; 
    try {
        dac.openStream( &parameters, NULL, RTAUDIO_FLOAT64,
                       sampleRate, &bufferFrames, &fbrp, (void *)&data );
        dac.startStream();
    }
    catch ( RtAudioError& e ) {
        e.printMessage();
        exit( 0 );
    }
    
    prompt1();
    
    //stop when over, but also when interrupted by pressing <enter> and ^C !

    bool b;
    
    struct timeval tv = get_duration (&data, (double)sampleRate);
    
    int ttyfd;
    
    if(STD){
        ttyfd = open("/dev/tty", O_RDWR);
        std::cout<<"fd: "<<ttyfd<<std::endl;
    }else
        ttyfd = STDIN_FILENO;
             
    b = rt_sig_timed_kbhit(ttyfd, tv, &data);
    
    if(0)std::cout<<"b = "<<b<<std::endl;
            
    if(b==1){
        std::cout<<"Playback stopped..."<<std::endl;
        data.STOP = true;
    }
    else
         std::cout<<"Playback finished..."<<std::endl;
        
    std::cout<<"Ending and quitting...\n"<<std::endl;
    if(STD) close(ttyfd);
   
    
    data.END = true;
    
    //wait for the stopped playback to die out (tail)
    if ((data.STOP)||(data.END))usleep(50000);
    
    if ( dac.isStreamRunning() == true )
        try {
        // Stop the stream    
            dac.stopStream();
        }
        catch (RtAudioError& e) {
            e.printMessage();
        }
    
cleanup:    
    if ( dac.isStreamOpen() ) dac.closeStream();
    return 0;
}

