//
//  rtmmain.cpp
//  rtm
//
//  Created by Stanko Juzbasic on 02/11/16.
//
//  this is a summarizing tutorial program, with updatable control data
//       and paralellization of the producer thread
//  written with use of RTaudio and RTmidi APIs by Gary Scavone
//  using a plain circular buffer

#include "RtAudio.h"
#include "RtMidi.h"
#include "cbuffer.h"
#include <iostream>
#include <iomanip>
#include <cstdlib>
#include <cmath>
#include <cstring>
#include <unistd.h>
#include <sys/sysctl.h>
#include <pthread.h>

# ifndef TWOPI
# define TWOPI 6.283185307179586476925287
# endif
# ifndef twopi
# define twopi TWOPI
# endif

#ifndef Byte
#define Byte char
#endif

#define MAXPART 4096 //maximum number of partials fixed to 4096 for so far

#define kBufferLength 2048

//audio proces state
#define idle 0
#define active 1
#define starting 2
#define stopping 3
#define stopped 4


//____declarations__________________________________________________________________
// globals
// Declare workThreadFarm, cond and mutex available to multiple threads
pthread_t       workThreadFarm;
pthread_cond_t  cond;
pthread_mutex_t mutex;

typedef unsigned int UInt32;

unsigned int sampleRate = 44100;

float*    Sin; //wavetable lookup

float     tuningA;

static int N = 0;

//user data argument
typedef struct {
    cbuf_t       cbuffer;     //circular buffer
    int          partials;    //number of partials
    unsigned int frames;      //number of frames
    float*  a;                //array of partial amplitudes
    float*  f;                //array of partial frequencies
    float*  preva;
    float*  prevf;
    int*    arg;              //array of accumulated phases (in terms of wavetable args)
    double  j;
    double  startingFrameCount;
    unsigned int   N;
    unsigned int totalFrames; //total number of synthesized frames
    int     state;
    bool    STOP;
    int     cpuNum;  // number of logical CPUs
    bool    on;      //a control flag
    float   vel;     //velocity
}uData;

//thread data argument
typedef struct tData{
    int tid;
    int cpuCount;
    uData* data;
}tData;

int   nThreads;

float **fbuffW;

int   numCand;

bool  debug;

RtMidiIn *midiin;

//MIDI information
bool    midi;
bool    omni;
int     hold;   //tracks how many notes have been held
int     prev_hold;
Byte    n[16];  //array of held MIDI notes
Byte    channel;//MIDI active channel number
float   pitchf; //read from MIDI

float   mmif; //midi modulation index (float) (0.-1.)
float   mxpf; //midi expression CC11  (float) (1.-4.)
Byte    bmi;  //midi byte debug value
float   changed;//modulation depth changed
float   changex;//expression ctrl. changed
float   bendf;  //bend value (float)
float   benddf; //bend depth (float)

//____________________________________________________________________________
#pragma mark - auxilliary function(s) -
//____________________________________________________________________________
void usage (void)
{
    std::cout << "\nUSAGE:\n";
    //std::cout << "rtm num_partials fund_frequency -[q|e|b]\n";
    std::cout << "rtm num_partials -[q|e|b]\n";
    //std::cout << "\t<fund_frequency> is in [Hz]\n";
    std::cout << "Options:   q   e   b   < =spectral envelope >\n";
    std::cout << "Example:\n";
    std::cout << "rtm 64 -b \n"<<std::endl;
    
    return;
}
//____________________________________________________________________________
void dproducer (bool on, uData* data)
{    
    if(debug)std::cout << "\tN = "<<N<<"\t Dummy Start"<<std::endl;
    return;
}
//____________________________________________________________________________
bool chooseMidiPort( RtMidiIn *rtmidi )
{
   std::string keyHit;
    
    if(0){
    std::cout << "\nWould you like to open a virtual input port? [y/N] ";
    std::getline( std::cin, keyHit );
    if ( keyHit == "y" ) {
        rtmidi->openVirtualPort();
        return true;
    }
    }
    std::string portName;
    std::cout << "\nRTMidi input port selection:" << std::endl;
    unsigned int i = 0, nPorts = rtmidi->getPortCount();
    if ( nPorts == 0 ) {
        std::cout << "No input ports available!" << std::endl;
        return false;
    }
    goto fix;
    if ( nPorts == 1 ) {
        std::cout << "\nOpening " << rtmidi->getPortName() << std::endl;
    }
    else {
        for ( i=0; i<nPorts; i++ ) {
            portName = rtmidi->getPortName(i);
            std::cout << "  Input port #" << i << ": " << portName << '\n';
        }
        
        do {
            std::cout << "\nChoose a port number: ";
            std::cin >> i;
        } while ( i >= nPorts );
        std::getline( std::cin, keyHit );  // used to clear out stdin
    }
fix:
    rtmidi->openPort( 0 );
    //rtmidi->openPort( i );
    
    return true;
}

//________________________________________________________________________________________
/*------------------------------------------------------------*/
/* PURPOSE: Evaluate Bessel function of first kind and order  */
/*          0 at input x  -written by M.G.R. Vogelaar         */
/*          Copyright (c) 1998  Kapteyn Institute Groningen   */
/*------------------------------------------------------------*/
double bessj0( double x )
{
    double ax,z;
    double xx,y,ans,ans1,ans2;
    
    if ((ax=fabs(x)) < 8.0) {
        y=x*x;
        ans1=57568490574.0+y*(-13362590354.0+y*(651619640.7
                                                +y*(-11214424.18+y*(77392.33017+y*(-184.9052456)))));
        ans2=57568490411.0+y*(1029532985.0+y*(9494680.718
                                              +y*(59272.64853+y*(267.8532712+y*1.0))));
        ans=ans1/ans2;
    } else {
        z=8.0/ax;
        y=z*z;
        xx=ax-0.785398164;
        ans1=1.0+y*(-0.1098628627e-2+y*(0.2734510407e-4
                                        +y*(-0.2073370639e-5+y*0.2093887211e-6)));
        ans2 = -0.1562499995e-1+y*(0.1430488765e-3
                                   +y*(-0.6911147651e-5+y*(0.7621095161e-6
                                                           -y*0.934935152e-7)));
        ans=sqrt(0.636619772/ax)*(cos(xx)*ans1-z*sin(xx)*ans2);
    }
    return ans;
}

//_____________________________________________________________________________
// generate m ramp samples 
double kmramp ( double i, int m)
{
    if(i >= m )   return 1.;
    if(i <= 0.)   return 0.;
    return .001*(i);
}
//_____________________________________________________________________________
float* makeSin (int rate)
{
    float* Sin = (float*)calloc(rate, sizeof(float));
    for(int i = 0; i < rate; i++)
        Sin[i] = sinf(TWOPI*i/(float)rate);
    return Sin;
}
//_____________________________________________________________________________
int getCPUnum (void)
{
    int numCPU;
    int mib[4];
    size_t len = sizeof(numCPU);
    
    // set the mib[] for hw.ncpu
    mib[0] = CTL_HW;
    mib[1] = HW_AVAILCPU;  // alternatively, try HW_NCPU;
    
    // get the number of logical CPUs from the system 
    sysctl(mib, 2, &numCPU, &len, NULL, 0);

    return numCPU;
}
//_____________________________________________________________________________
// synthesis algorithm using wavetable lookup - thread
//_____________________________________________________________________________
void *producerTN  (void *TN)
{
    float b;
    int     arg;
    float samples[4096];
    tData    threadData;
    threadData = *((tData *)TN);
    int tid  = threadData.tid;
    int step = threadData.cpuCount;
    int *ret = (int*)calloc(1,sizeof(int));
    int rate       = (int) sampleRate;
    float frames = (float)threadData.data->frames;
    
    if(debug) std::cout<<"Thread#: "<<tid<<"\nNumCPU = "<<step<<std::endl;

    for(int k = tid; k < threadData.data->partials; k = k + step){
        if(debug) std::cout<<"K = "<<k<<"\t prev.amp = "<<threadData.data->preva[k]<<std::endl;
        if(k > threadData.data->partials)break;
        
        if(k >= numCand)
            threadData.data->a[k] = 0.;
        //ampl. and phase control values' integration for possible GUI updates
        float da = (threadData.data->a[k] - threadData.data->preva[k])/frames;        
        float deltaf = (threadData.data->f[k] - threadData.data->prevf[k]);
        float df     = deltaf/frames;
        int idf        = (int)df;
        int dph;
        arg = threadData.data->arg[k];
        dph = (int)(threadData.data->prevf[k]);
        
        for (UInt32 frame = 0; frame < threadData.data->frames; frame++){
            dph += idf;
            arg += dph;
            if(arg > rate)arg %= rate;
            b = .2*Sin[arg];
            samples[frame]  += b;
            samples[frame]  *= (threadData.data->preva[k] +frame*da);
        }//end frame
        threadData.data->prevf[k] = threadData.data->f[k];
        threadData.data->preva[k] = threadData.data->a[k];
        threadData.data->arg[k] = arg;
        
        for (int i = 0; i < threadData.data->frames; i++)fbuffW[tid][i] += samples[i];
    }//end k
    
    return (void*)ret;
}
//_____________________________________________________________________________
//  threadable producer routine:
//      segments the code into 'num' parallel threads,
//      computed from the number of logical CPU cores
//_____________________________________________________________________________
void *producerPSum(void* pdata)
{
    int rc;
    uData*    data;
    data = ((uData *)pdata);
    int cpuNum = data->cpuNum;
    bool on = data->on;
    int *retSum = (int*)calloc(1,sizeof(int));
    pthread_t threads[cpuNum];
    tData thread_args[cpuNum];
    void* resulT;
    static float frames  [4096];
    int     bytesToCopy, bytes;
    int32_t availableBytes;
    
    
    while(on){
        if(data->STOP) data->state = stopping;

        float xp = (pitchf - 69.)/12.;   
        float f = tuningA*powf(2.,xp);
            
        if(data->f[0] != f){
            if(debug)std::cout<<"note = "<<pitchf<<"\tf = "<<f<<std::endl;
            for(int i = 0; i < data->partials; i++){
                data->f[i] = (i + 1)*f;
                }
            }
        
        bytesToCopy = data->frames*sizeof(float);
        memset(frames,  0, 4096*sizeof(float));
        availableBytes = 0;

        
        if( (fbuffW  = (float**)calloc(cpuNum + 1, sizeof(float*)))!= NULL)
            for (int i=0; i<cpuNum; ++i)
            {
                fbuffW[i] = (float*)calloc(data->frames, sizeof(float));
                thread_args[i].tid = i; //ord. number of thread
                thread_args[i].cpuCount = cpuNum; //counter increment step
                thread_args[i].data = data;
                rc = pthread_create(&threads[i], NULL, producerTN, (void *) &thread_args[i]);
                if(rc != 0) std::cerr<<"rc = "<<strerror(rc)<<std::endl;
            }
        
        for (int i=0; i<cpuNum; ++i) rc = pthread_join(threads[i], &resulT);
        
        for(UInt32 samp = 0; samp < data->frames; samp++)
            for(int i = 0; i < cpuNum; i++)
                frames[samp] += fbuffW[i][samp];
        
        switch (data->state) { //graceful suspension and termination...
            case idle:    memset(frames, 0, data->frames*sizeof(float)); break;
            case stopped: memset(frames, 0, data->frames*sizeof(float));data->state = idle; break;
            case starting:
                for(UInt32 samp = 0; samp < data->frames; samp++)frames[samp] *= (float)kmramp(N++,data->frames);
                data->state = active; break;
            case stopping:
                for(UInt32 samp = 0; samp < data->frames; samp++)frames[samp] *= (float)kmramp(N--,data->frames);
                if (N <= 0){N = 0; data->state = stopped;}
                break;
            default: break;
        }
       blah: 

        bytes = bytesToCopy;
        int numBytes = cbuf_offer(&(data->cbuffer), (unsigned char*)frames, bytes);
        numBytes = 0;
            
        pthread_cond_wait (&cond, &mutex);
        
        for(int i = 0; i < cpuNum; i++)
           if(fbuffW[i]) free(fbuffW[i]);
        if (fbuffW)free(fbuffW);
    } //end while(on)    
    return (void*)retSum;
}
//___________________________________________________________________________________
// worker thread launcher - for all producer routines
//___________________________________________________________________________________
pthread_t pdproducer (bool on, uData* data)
{
    pthread_t sumThread;
    data->on = on;
    data->state = idle;
    int rcPsum;
    
    rcPsum = pthread_create(&sumThread, NULL, producerPSum, (void *) data);
    return sumThread;
}
//____________________________________________________________________________________
#pragma mark - callback -
//____________________________________________________________________________________
// based on example callback by Gary Scavone
// rewritten by SJ for monophonic controller, "hacked" note-off catching and thread farm
void midiCB( double deltatime, std::vector< unsigned char > *message, void *userData )
{
    //message is an array of size = message->size
    //fields are accessed through function (int)message->at(i)
    
    uData *gen = (uData*)userData;
    if(debug)std::cout << "f0 = " << gen->f[0] << std::endl;
    
    unsigned int nBytes = message->size();
    int midiStatus, midiCommand, midiChannel, note, vel, ctrl, val, valm, vall;
    float bendf;
    if(0){ // for debugging suspectedly broken MIDI interface
    for ( unsigned int i=0; i<nBytes; i++ )
        if (nBytes > 1 )
            std::cout << "Byte " << i << " = " << (int)message->at(i) << ", ";
    if ( nBytes > 1 ) //if ( nBytes > 0 )
        std::cout << "stamp = " << deltatime << std::endl;
    } //end if(0)
    
    for ( unsigned int i=0; i<nBytes; i++ )
        if (nBytes > 1 ){
    if(i == 0){
        midiStatus  = (int)message->at(i);
        midiChannel = midiStatus & 0x0F;
        midiCommand = midiStatus >> 4;
        //std::cout<<"Ch: 0x"<<std::hex<<midiChannel<<", cmd: 0x"<<std::hex<<midiCommand <<", ";
    }
            
    if(i == 1){
        if(midiCommand == 0x8){
        note = (int)message->at(i);
        //std::cout << "Note: "<< note<<", ";
        }
        if(midiCommand == 0x9){
        note = (int)message->at(i);
        //std::cout << "Note: "<< note<<", ";
        }
        if(midiCommand == 0xB){ //CC
        ctrl = ((int)message->at(i) & 0x7F);
        //std::cout << "Ctrl: "<< ctrl<<", ";
        }
        if(midiCommand == 0xE){
        //std::cout << "Pitch Bend: ";
        vall = ((int)message->at(i) & 0x7F);
        }
    }
    if(i == 2){
        if(midiCommand == 0x9){
        vel = (int)message->at(i);
        //std::cout << "Velocity: "<< vel<<" ";
        }
        if(midiCommand == 0xB){ //CC (01 - mod, 11 - Xpr, 40 - footsw)
        val = ((int)message->at(i) );
        if(ctrl == 1 )mmif = (float)val/127.; //0 - 1
        if(ctrl == 11)mxpf = 3*(float)val/127. + 1 ; //1 - 4
        //std::cout << "Value: "<< val<<", ";
        }    
        if(midiCommand == 0xE){ //Pitch
        valm = ((int)message->at(i) & 0x7F);
        bendf = 128*(float)valm + (float)vall - 8192.;
        bendf = bendf/8192.;
        //std::cout << "Value: "<< bendf<<" ";
        }
      }
    }//end if(nBytes > 1)...
    if ( nBytes > 1 ){ //if ( nBytes > 0 )
        if(debug){std::cout << "stamp = " << deltatime;
        std::cout << std::endl;
        }
    }
    //monophonic controller algorithm---------
    if(1)
    if(midiCommand != 0xF)
    {
        //debug
        if(debug)std::cout<<"Channel: "<<midiChannel<<", Message: "<<midiCommand<<std::endl;
        if (!omni)
            if(midiChannel != channel)
                goto next_packet;
        
        if(midiCommand == 0x09)
        {
        //    note = packet->data[1] & 0x7F;
        //    velocity = packet->data[2] & 0x7F;
            if(0)std::cout<<"Note: "<<note<<std::endl;
            
            //!!!hacked note-off!!!
            if (vel == 0x0){
                //'note' is the note released!!
                if( hold > 0){//find which held note has been released!
                    for(int i = 1; i <= hold; i++) //get the note and its ord. number
                        if(note == n[i]){ //last note released
                            if (i == hold) {
                                goto h;
                            }
                            //any other note released ?
                            //compact 'n' array, keep last note's highest position
                            for (int j = i+1; j <= hold; j++)
                                n[j-1] = n[j];
                        }
                h:
                    if(hold > 0){
                        note = n[hold-1];

                    }//1
                    if(hold > 1)//prevents note = 0
                        pitchf = (float)note + bendf;

                    prev_hold = hold;//089
                    hold--;//this enables last note to be heard
                }
                //note on!!!
            }else{                
                if(debug)std::cout<<"Vel = "<<(int)vel<<std::endl;
                prev_hold = hold;//089
                hold++;
                if(debug)std::cout<<"Note "<<note<<" ON, hold = "<< hold<<std::endl;

                pitchf = (float)note + bendf;
                
                if((hold == 1)&&(prev_hold == 0)){
                    if(gen->state == idle){
                        gen->on = true; 
                        gen->state = starting;
                        dproducer(true, gen);
                    }else if (gen->state == stopped)gen->state = starting;
                    else if  (gen->state == stopping)gen->state = active;
                    if(0)std::cout<<"||||||||||||||||||"<<std::endl;
                }
                n[hold] = note;
            }
            if(0)std::cout<<"HOLD = "<<hold<<std::endl;
            
            if(hold ==  0){
                if(gen->state == active){; gen->state = stopping;}
                //on = false;
            }
        }
        
        //!!!proper note-off!!!
        if(midiCommand == 0x08)
        {
            //note = packet->data[1] & 0x7F;
            if( hold > 0){ //find which note has been released!
                for(int i = 1; i <= hold; i++)
                    if(note == n[i]){
                        if (i == hold) {
                            goto hh;
                        }
                        for (int j = i+1; j <= hold; j++)
                            n[j-1] = n[j];
                    }
            hh:
                if(hold > 0){
                    note = n[hold-1];
                }
                if(hold > 1)//prevents note = 0
                    pitchf = (float)note + bendf;
                
                prev_hold = hold;//089
                hold--;
                
            }
            if(hold ==  0){
                if(gen->state == active){; gen->state = stopping;}
            }            
        }
        //0xB -control change: "1011"
        if(midiCommand == 0xB){
            //Byte ctrl = packet->data[1] & 0x7F;
            if(ctrl == (0x01 & 0x7F)){          //C01 -> mod. depth
                //Byte val  = packet->data[2] & 0x7F;
                bmi  = val;
                mmif = (float)val/127.   ;
                changed = TRUE;
                if(debug)std::cout<<"mod:"<<mmif<<std::endl;
            }
            if (ctrl == (0x0B & 0x7F)){         //C11 -> expression
                //Byte val  = packet->data[2] & 0x7F;
                bmi = val;
                mxpf = 3*(float)val/127. + 1 ;
                changex = TRUE;
                if(debug)std::cout<<"xpr:"<<mxpf<<std::endl;
            }
        }
        //pitch bend: 0xE
        if(midiCommand == 0xE){
            pitchf = (float)note + bendf;
            if(debug)std::cout<<"pbend = "<<bendf<<std::endl;
            if(debug)std::cout<<"3Note:  "<<note<<std::endl;
        }
        
    next_packet:
        if (hold == 0) {
            if(prev_hold != 0 ){
              prev_hold = 0;//089
                if(0)std::cout<<"hold = "<<hold<<"\tprev_hold = "<<prev_hold<<std::endl;;
            }
            ;
        }
    }
    
jmp: ;
        
}  // end SJ
//____________________________________________________________________________________
// RtAudio callback reading from circular buffer, with start/stop functionality
int cbrtc(void *outputBuffer,
         void *inputBuffer,
         unsigned int nBufferFrames,
         double streamTime,
         RtAudioStreamStatus status,
         void *userData )
{

    double *o_buffer = (double *) outputBuffer;
    
    float b_buffer [1024];

    uData *data = (uData*)userData;
    
    pthread_mutex_lock(&mutex);
    
    unsigned int frame;
    double  d;

    if ( status )
        std::cout << "Stream underflow detected!" << std::endl;

    int used = cbuf_usedspace(&(data->cbuffer));
    unsigned char *ptr0 = cbuf_peek(&(data->cbuffer));
    if(ptr0 != NULL)memcpy(b_buffer, (const void*)ptr0, used);
    cbuf_poll(&(data->cbuffer), used);

    
    // Writing interleaved stereo output buffer
    for (frame = 0; frame < nBufferFrames; ++frame)
	{
        d = 0.;
        d = b_buffer[frame];
        //start & stop de-glitching;
        if(data->totalFrames < data->N) d = d*kmramp(data->totalFrames, nBufferFrames);
        if(data->STOP)d = d*(data->N > 0 ? kmramp(data->N--, nBufferFrames): 0.);
        
        o_buffer[2*frame+1] = o_buffer[2*frame] = d;

		data->j += 1.0;
        data->totalFrames +=1;
	}    
    data->frames = nBufferFrames;
    
    pthread_cond_signal (&cond);
    pthread_mutex_unlock (&mutex);
    
    return 0;
}
//____________________________________________________________________________________
#pragma mark - main -
//____________________________________________________________________________________
int main(int argc, const char * argv[])
{
    int i = 1;
    int option = 1;
    int fe; //frequency envelope
    double temp = 0.;
    int num = getCPUnum();
    double f0 = 65.31;
    debug = false;    
    uData cData;
    workThreadFarm = 0;
    tuningA = 440.;
    
    if(debug)    std::cout << argc <<"\n";    
    if(argc < 2 || argc > 3){
    u:
        usage();
        return 0;
    }    
    //read number of partials
    cData.partials = atoi(argv[1]);
    if (cData.partials == !cData.partials)goto u;
    if (cData.partials < 1)goto u;

    fe = 1;
    if(argc == 3){
    if(atof(argv[2]) == 0.)option = 1;
    if(strncmp(argv[2],"-q",2) == 0)
    {option = 1; fe = 2;}
    if(strncmp(argv[2],"-e",2) == 0)
    {option = 1; fe = 3;}
    if(strncmp(argv[2],"-b",2) == 0)
    {option = 1; fe = 4;}
    }
    if(0){std::cout << "option "<<fe<<"\n"; exit(-1); }
    std::cout << "\n_______________________________________________\n";
    std::cout << "* Threaded tonegenerator for RTAudio & RTMidi *\n";
    std::cout << "- - - -  running on "<<num<<" logical CPU cores - - - -\n";
    std::cout << "\nRendering " <<cData.partials<< " partial"<<(cData.partials > 1 ? "s ":" ");
    std::cout << "of a " <<f0<<" Hz fundamental\n\n";
    
    //data arranging and preparation
    //allocate all the necessary cData arrays
    cData.f          = (float*)calloc(cData.partials, sizeof(float));
    cData.a          = (float*)calloc(cData.partials, sizeof(float));
    cData.prevf      = (float*)calloc(cData.partials, sizeof(float));
    cData.preva      = (float*)calloc(cData.partials, sizeof(float));
    cData.arg        = (int*)calloc(cData.partials, sizeof(int));
    
    numCand = cData.partials > MAXPART ? MAXPART : cData.partials;
    
    //construct partial frequencies
    for(i = 1; i < cData.partials; i++) {
        cData.f[i-1] = f0*i;
        if(cData.f[i-1] <0.) cData.f[i-1] *= -1.;
        if (cData.f[i-1] > 20000.) cData.f[i-1] = 20000.;
        if(debug) std::cout<<cData.f[i-1]<<"\n";
    }
    
    //make sine table    
    Sin = makeSin(sampleRate);

    //partialAmplitudes' spectral evnelope (=reciprocal, quadratic, bessel, exponential)
    temp = 0;
    
    for(i = 0; i < cData.partials; i++ ){
        if (fe == 4) temp += fabs((float)bessj0((double)i));
        if (fe == 3) temp += exp(-.5*i);
        if (fe == 2) temp += 1./((float)(i*i + 2*i + 1));
        if (fe == 1) temp += 1./(float)(i+1);
    }
    temp = 1./temp;
    if(debug) std::cout <<"temp = "<<temp<<"\n";

    for(i = 0 ; i < cData.partials; i++){
        if (fe == 1) cData.a[i] = .3*temp/(float)(i+1);
        if (fe == 2) cData.a[i] = .3*temp/((float)(i*i + 2*i + 1));
        if (fe == 3) cData.a[i] = .3*temp*exp(-.5*i);
        if (fe == 4) cData.a[i] = .01*(float)(bessj0((double)(i-2))*bessj0((double)(i-2)))/temp;
    }
    
    if(debug)for(i = 0; i < cData.partials; i++)std::cout <<i+1<<"\tfreq = "<<cData.f[i]<<"\tamp = "<<cData.a[i] <<"\n";
    
    if (fe == 1)std::cout << "\nLinear ";
    if (fe == 2)std::cout << "\nQuadratic ";
    if (fe == 3)std::cout << "\nExponential ";
    if (fe == 4)std::cout << "\nBessel J0 ";
    std::cout << "spectral evnelope\n\n";
    
    cData.STOP = false;
    cData.N = 1000;
      
    // Initialize circular buffer
    cData.cbuffer = *cbuf_new(12);
    
    //RtAudio consumer thread setup and call
    RtAudio dac;
    if ( dac.getDeviceCount() < 1 ) {
        std::cout << "\nNo audio devices found!\n";
        exit( 0 );
    }
    RtAudio::StreamParameters parameters;
    parameters.deviceId = dac.getDefaultOutputDevice();
    parameters.nChannels = 2;
    parameters.firstChannel = 0;
    cData.totalFrames = 0;
    unsigned int bufferFrames = 256; // 256 sample frames
    cData.frames = bufferFrames;
    cData.cpuNum = num;
    
    //create the mutex and cond to coordinate between producer and consumer threads
    pthread_cond_init  (&cond, NULL);
    pthread_mutex_init (&mutex, NULL);
    
    cData.j = 0.;
    try {
        dac.openStream( &parameters, NULL, RTAUDIO_FLOAT64,
                       sampleRate, &bufferFrames, &cbrtc, (void *)&cData );
        dac.startStream();
    }
    catch ( RtAudioError& e ) {
        e.printMessage();
        exit( 0 );
    }
    
    midiin = 0;
    try{       
        midiin = new RtMidiIn();
        if ( chooseMidiPort( midiin ) == false ) goto cleanup;
        // Set MIDI callback function.  This should be done immediately after
        // opening the port to avoid having incoming messages written to the
        // queue instead of sent to the callback function.
        // important is to also register userData here, else MIDI callback won't work!
        midiin->setCallback( &midiCB, (void *)(&cData) );
        
        // Don't ignore sysex, timing, or active sensing messages.
        midiin->ignoreTypes( false, false, false );
        //preset userData state
        cData.state = idle;
        
        //create producer thread farm
        workThreadFarm = pdproducer(true, &cData);     
        std::cout<<"Work Thread Farm ID = "<< (long)workThreadFarm <<std::endl;
        
    } catch ( RtMidiError &error ) {
        error.printMessage();
    }
    
    char input;
    std::cout << "\nPlaying ... press <enter> to quit."<<std::endl;
    std::cin.get( input );
    
    cData.STOP = true;
    usleep(40000);

    pthread_mutex_unlock(&mutex);
    pthread_mutex_destroy(&mutex);
    pthread_cond_destroy(&cond);
   
    try {// Stop the stream
        dac.stopStream();
    }
    catch (RtAudioError& e) {
        e.printMessage();
    }

    if ( dac.isStreamOpen() ) dac.closeStream();
    std::cout << "\nQuitting...";    
    if(pthread_cancel(workThreadFarm) == 0)std::cout << "   ...success"<<std::endl;
    workThreadFarm = 0;
    std::cout << (long)workThreadFarm<<std::endl;
cleanup:
    delete midiin;
    return 0;
}
