//
//  rtswg.cpp
//  rtswg
//
//  Created by Stanko Juzbasic on 1/19/16.
//
//  this is the Sine Wave Generator program, written in RTaudio

#include "RtAudio.h"
#include <iostream>
#include <cstdlib>
#include <cmath>
#include <cstring>
#include <unistd.h>
# ifndef TWOPI
# define TWOPI 6.283185307179586476925287
# endif

//____declarations_______________________________________________________________________
// globals
unsigned int sampleRate = 44100;
double       rad;

typedef struct {
    int     partials;
    double* partialFrq;
    double* partialAmp;
    double  j;
    double  startingFrameCount;
    unsigned int   N;
    unsigned int totalFrames;
    bool     STOP;
}uData;

//Prototypes______________________________________________________________________________

double bessj0 (double);

double kramp(double);

void   usage (void);
//________________________________________________________________________________________
#pragma mark - auxilliary function(s) -
//________________________________________________________________________________________
void usage (void)
{
    std::cout << "\nUSAGE:\n";
    std::cout << "rtswg frq1 frq2 frq3 ... frqn -[q|e|b]\n";
    std::cout << "\t<frequencies> are in [Hz]\n";
    std::cout << "Options:   q   e   b   < =spectral envelope >\n";
    std::cout << "Example:\n";
    std::cout << "rtswg 220. 330. 550. 990. \n\n";
    
    return;
}
//________________________________________________________________________________________
/*------------------------------------------------------------*/
/* PURPOSE: Evaluate Bessel function of first kind and order  */
/*          0 at input x  -written by M.G.R. Vogelaar         */
/*          Copyright (c) 1998  Kapteyn Institute Groningen   */
/*------------------------------------------------------------*/
double bessj0( double x )
{
    double ax,z;
    double xx,y,ans,ans1,ans2;
    
    if ((ax=fabs(x)) < 8.0) {
        y=x*x;
        ans1=57568490574.0+y*(-13362590354.0+y*(651619640.7
                                                +y*(-11214424.18+y*(77392.33017+y*(-184.9052456)))));
        ans2=57568490411.0+y*(1029532985.0+y*(9494680.718
                                              +y*(59272.64853+y*(267.8532712+y*1.0))));
        ans=ans1/ans2;
    } else {
        z=8.0/ax;
        y=z*z;
        xx=ax-0.785398164;
        ans1=1.0+y*(-0.1098628627e-2+y*(0.2734510407e-4
                                        +y*(-0.2073370639e-5+y*0.2093887211e-6)));
        ans2 = -0.1562499995e-1+y*(0.1430488765e-3
                                   +y*(-0.6911147651e-5+y*(0.7621095161e-6
                                                           -y*0.934935152e-7)));
        ans=sqrt(0.636619772/ax)*(cos(xx)*ans1-z*sin(xx)*ans2);
    }
    return ans;
}

//___________________________________________________________________________________
// generate 1000 ramp samples 
double kramp ( double i)
{
    if(i >= 1000.)return 1.;
    if(i <= 0.)   return 0.;
    return .001*(i);
}
//____________________________________________________________________________________
#pragma mark - callback -
//____________________________________________________________________________________
// Two-channel sine wave generator with arbitrary number of partials and start/stop
int swrp(void *outputBuffer,
         void *inputBuffer,
         unsigned int nBufferFrames,
         double streamTime,
         RtAudioStreamStatus status,
         void *userData )
{

    double *buffer = (double *) outputBuffer;

    uData *data = (uData*)userData;
    
    unsigned int i, frame;
    double  d, p;
    double two_pi_norm = TWOPI/sampleRate;
    
    
    if ( status )
        std::cout << "Stream underflow detected!" << std::endl;
    
    // Writing interleaved stereo audio data.
    for (frame = 0; frame < nBufferFrames; ++frame)
	{
        d = 0.;
        // additive synthesis of partials
        for(i = 0; i < data->partials; i++){
            p =  sin (two_pi_norm * data->j*(data->partialFrq[i]));
            p = p*data->partialAmp[i];
            d = d+p;
        }
        
        //start & stop de-glitching;
        if(data->totalFrames < data->N) d = d*kramp(data->totalFrames);
        if(data->STOP)d = d*(data->N > 0 ? kramp(data->N--): 0.);
        
        buffer[2*frame+1] = buffer[2*frame] = d;

		data->j += 1.0;
        data->totalFrames +=1;
	}    
    
    return 0;
}
//____________________________________________________________________________________
#pragma mark - main -
//____________________________________________________________________________________
int main(int argc, const char * argv[])
{
    int i = 1;
    int j = 0;
    int option;
    int fe;
    double temp = 0.;
    uData data;
    
    if(argc < 2){
        usage();
        return 0;
    }
    if(0)    std::cout << argc <<"\n";
    
    i = argc - 1;
    fe = 1;
    if(atof(argv[i]) == 0.)option = 1;
    if(strncmp(argv[i],"-q",2) == 0)
    {option = 1; fe = 2;}
    if(strncmp(argv[i],"-e",2) == 0)
    {option = 1; fe = 3;}
    if(strncmp(argv[i],"-b",2) == 0)
    {option = 1; fe = 4;}
    
    if(0){
    std::cout << "option "<<fe<<"\n";
            exit(-1);
    }
    std::cout << "\n_____________________________________________\n";
    std::cout << "* * *  Sine Wave Generator for RTAudio  * * *\n";
    std::cout << "\nRendering " <<argc - option - 1<< " partial"<<(argc - option > 2 ? "s":" ")<<"\n\n";
    
    //user data arranging and  preparation
    //allocate freq. and amp. arrays
    data.partialFrq      = (double*)calloc(argc-option, sizeof(double));
    data.partialAmp      = (double*)calloc(argc-option, sizeof(double));
    
    //read freq. arguments
    for(i = 1; i < argc - option; i++) {
        data.partialFrq[i-1] = atof(argv[i]);
        if(data.partialFrq[i-1] <0.) data.partialFrq[i-1] *= -1.;
        if (data.partialFrq[i-1] > 20000.) data.partialFrq[i-1] = 20000.;
        if(0) std::cout<<data.partialFrq[i-1]<<"\n";
    }
 
    //read the number of partials
    data.partials = argc - option - 1;
    
    //sort frequencies in ascending order   
    for(i = 0; i < data.partials - 1; i++)
    {
        for (j = 0 ; j < data.partials - i - 1; j++)
        {
            if (data.partialFrq[j] > data.partialFrq[j+1])
            {
                temp       = data.partialFrq[j];
                data.partialFrq[j]   = data.partialFrq[j+1];
                data.partialFrq[j+1] = temp;
            }
        }
    }

    //partialAmplitudes spectral evnelope (1=reciprocal, quadratic, exponential, bessel)
    temp = 0;
    
    for(i = 0; i < data.partials; i++ ){
        if (fe == 4) temp += fabs((float)bessj0((double)i));
        if (fe == 3) temp += exp(-.5*i);
        if (fe == 2) temp += 1./((float)(i*i + 2*i + 1));
        if (fe == 1) temp += 1./(float)(i+1);
    }
    temp = 1./temp;
    if(0) std::cout <<"temp = "<<temp<<"\n";
    for(i = 0 ; i < data.partials; i++){
        if (fe == 1) data.partialAmp[i] = .3*temp/(float)(i+1);
        if (fe == 2) data.partialAmp[i] = .3*temp/((float)(i*i + 2*i + 1));
        if (fe == 3) data.partialAmp[i] = .3*temp*exp(-.5*i);
        if (fe == 4) data.partialAmp[i] = .04*(float)(bessj0((double)(i-2))*bessj0((double)(i-2)))/temp;
    }
    for(i = 0; i < data.partials; i++)
        std::cout <<i+1<<"\tfreq = "<<data.partialFrq[i]<<"\tamp = "<<data.partialAmp[i] <<"\n";
    
    if (fe == 1)std::cout << "\nLinear ";
    if (fe == 2)std::cout << "\nQuadratic ";
    if (fe == 3)std::cout << "\nExponential ";
    if (fe == 4)std::cout << "\nBessel J0 ";
    std::cout << "spectral evnelope\n\n";
    
    data.STOP = false;
    data.N = 1000;
    
    //actual synthesis setup and call
    RtAudio dac;
    if ( dac.getDeviceCount() < 1 ) {
        std::cout << "\nNo audio devices found!\n";
        exit( 0 );
    }
    RtAudio::StreamParameters parameters;
    parameters.deviceId = dac.getDefaultOutputDevice();
    parameters.nChannels = 2;
    parameters.firstChannel = 0;
    rad = TWOPI/sampleRate;
    data.totalFrames = 0;
    unsigned int bufferFrames = 256; // 256 sample frames
    data.j = 0.; 
    
    // pointer to callback function swrp() is registered in the call
    try {
        dac.openStream( &parameters, NULL, RTAUDIO_FLOAT64,
                       sampleRate, &bufferFrames, &swrp, (void *)&data );
        dac.startStream();
    }
    catch ( RtAudioError& e ) {
        e.printMessage();
        exit( 0 );
    }
    
    char input;
    std::cout << "\nPlaying ... press <enter> to quit.\n";
    std::cin.get( input );
    data.STOP = true;
    usleep(40000);
    
    std::cout<<"Stopping and quitting...\n";
    
    try {
        // Stop the stream    
        dac.stopStream();
    }
    catch (RtAudioError& e) {
        e.printMessage();
    }
    if ( dac.isStreamOpen() ) dac.closeStream();
    
    return 0;
}

